unit Dscomp;
{
  TSourceList, Enhanced TStringList class
  Copyright (c) 1996, by Philip Stevenson
}

interface

uses
  {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  SysUtils, Classes, Dialogs, Controls,
  FileInfo;

type

  TSourceList = class(TStringList)
  private
    { Private declarations }
    FTag: Byte;
    FRange: Smallint;      {used for scan}
    FMaxRange: Integer;     {"}
    FPosition: Integer;     {"}
    FUpAhead: Integer;      {"}
    FEndPos: Integer;       {"}
    FBeyondRange: boolean;  {"}
    FBeyondEnd: boolean;    {"}
    FFirstMismatch: Integer;  {"}
    FLastMismatch: Integer;  {"}
    FSourceFile: string; { name}
    FSourcePath: string; { and path }
    FCreated: boolean;
    function GetFirstMismatch: string;
    function GetLastMismatch: string;
    procedure SetSourceFile(const Value: string);
    function GetStr(Apos: Integer; const N, limit: Integer): string;
  public
    { Public declarations }
    constructor Create(Atag: Byte);

    procedure SetRange;
    function StrAtPos: string;
    function StrAtOffset(const N: Smallint): string;
    function StrUpNext(const N: Smallint): string;
    procedure NextPos;
    procedure PrevPos;
    procedure NextUpAhead;
    procedure ClearPositions;
    procedure LoadList;
    function DeleteCreatedFile: string;
    class procedure DFMtoTXT(const dfmfile, txtfile: string);

    property SourceFile: string read FSourceFile write SetSourceFile;
    property FirstMismatch: string read GetFirstMismatch;
    property LastMismatch: string read GetLastMismatch;
    property UpAhead: Integer read FUpAhead;
    property Position: Integer read FPosition;
    property SourcePath: string read FSourcePath;
    property Tag: Byte read FTag;
    property BeyondRange: boolean read FBeyondRange;
    property BeyondEnd: boolean read FBeyondEnd;
    property EndPos: Integer read FEndPos;
  end;


  function IsBlank(const S: string): boolean;
  function TimeDiffStr(const dtime: TDateTime): string;
  function IsFormFile(const FN: string): boolean;
  function Get2ndFileName(const FN: string): string;

implementation

const
   MAX_SCAN_AHEAD = 50; {Max lines to scan ahead on mismatch}
   ERR_MSG = 'Out-of-range SourceList index: %d.';

  function IsBlank(const S: string): boolean;
  {-Return True for empty having only space or ctrl chars}
  var
    ix: Integer;
  begin
    Result := True;
    for ix := 1 to Length(S) do
      if S[ix] > ' ' then
      begin
        Result := False;
        exit;
      end
  end;

  function TimeDiffStr(const dtime: TDateTime): string;
  {-Build string based on time difference}
  var
    stime: TDateTime;
  begin
    if dtime >= 1.0 then
      Result := Format('%3.1f days', [dtime])
    else
    begin
      stime := dtime * 24;
      if stime < 1.0 then
        Result := Format('%2.1f minutes', [stime*60])
      else
        Result := Format('%2.1f hours', [stime])
    end
  end;

  function GetBakExt(const FN: string): string;
  {-Get bakup file name for editable Delphi file}
  const
    PAIRS = 8;
    ExtPairs: array[1..2, 1..PAIRS] of string[4] =
    (('.pas', '.dpr', '.dfm', '.txt', '.dop', '.opt', '.inc', '.rc'),
     ('.~pa', '.~dp', '.~df', '.~tx', '.~do', '.~op', '.~in', '.~rc'));
  var
    ic, ir: Shortint;
  begin
    for ir := 1 to 2 do
      for ic := 1 to PAIRS do
        if CompareText(ExtractFileExt(FN), ExtPairs[ir, ic]) = 0 then
        begin
          Result := ExtPairs[3-ir, ic];
          exit
        end;
    Result := '';
  end;

  function IsFormFile(const FN: string): boolean;
  var
    S: string;
  begin
    S := ExtractFileExt(FN);
    Result := (CompareText(S, '.dfm') = 0) or
     (CompareText(S, '.~df') = 0);
  end;

  function Get2ndFileName(const FN: string): string;
  {-Get 2nd file name }
  var
    BakExt, FileName: string;
  begin
    Result := '';
    BakExt := GetBakExt(FN);
    if BakExt = '' then
      exit;
    FileName := ChangeFileExt(FN, BakExt);
    if FileExists(FileName) then
      Result := FileName;
  end;

  {TSourceList methods}

  constructor TSourceList.Create(Atag: Byte);
  begin
    inherited Create;
    FTag := Atag;
    FMaxRange := MAX_SCAN_AHEAD; {lines}
  end;

  procedure TSourceList.NextPos;
  begin
    if FBeyondEnd then
      raise EStringListError.CreateFmt(ERR_MSG, [FPosition]);
    inc(FPosition);
    FBeyondEnd := FPosition > FEndPos;
  end;

  procedure TSourceList.PrevPos;
  begin
    if FPosition <= 0 then
      raise EStringListError.CreateFmt(ERR_MSG, [FPosition]);
    dec(FPosition);
    FBeyondRange := False;
  end;

  procedure TSourceList.NextUpAhead;
  begin
    if FBeyondRange then
      raise EStringListError.CreateFmt(ERR_MSG, [FUpAhead]);
    inc(FUpAhead);
    if FUpAhead > FRange then
      FBeyondRange := True;
  end;

  procedure TSourceList.SetRange;
  begin
    if FFirstMismatch = 0 then
      FFirstMismatch := FPosition+1;
    FLastMismatch := FPosition+1;
    FBeyondRange := False;
    FUpAhead := FPosition;
    FRange := FMaxRange+FUpAhead;
    if FRange > FEndPos then
      FRange := FEndPos;
    NextUpAhead;
  end;

  procedure TSourceList.ClearPositions;
  begin
    FFirstMismatch := 0;
    FLastMismatch := 0;
    FPosition := 0;
    FBeyondEnd := False;
    FBeyondRange := False;
  end;

  function TSourceList.GetFirstMismatch: string;
  begin
    Result := IntToStr(FFirstMismatch);
  end;

  function TSourceList.GetLastMismatch: string;
  begin
    Result := IntToStr(FLastMismatch);
  end;

  function TSourceList.GetStr(Apos: Integer; const N, limit: Integer): string;
  {-Get string at Position; return null str if looking beyond end}
  begin
    if N > 0 then
    begin
      inc(Apos, N);
      if Apos > FEndPos then
      begin
        Result := '';
        exit;
      end
    end;
    if FBeyondEnd or (Apos < 0) then
      raise EStringListError.CreateFmt(ERR_MSG, [Apos]);
    Result := Strings[Apos]
  end;

  function TSourceList.StrAtOffset(const N: Smallint): string;
  {-Get string at Position}
  begin
    Result := GetStr(FPosition, N, FEndPos)
  end;

  function TSourceList.StrAtPos: string;
  {-Get string at Position}
  begin
    Result := GetStr(FPosition, 0, FEndPos)
  end;

  function TSourceList.StrUpNext(const N: Smallint): string;
  {-Get string up ahead}
  begin
    Result := GetStr(FUpAhead, N, FRange)
  end;

  procedure TSourceList.LoadList;
  {-Load file contents to SourceList}
  begin
    try
      LoadFromFile(FSourceFile);
      FEndPos := Count-1;
    except
      raise EInOutError.Create('Cannot load file: '+FSourceFile+#13+
      ' to source list for processing.')
    end;
  end;

  class procedure TSourceList.DFMtoTXT(const dfmfile, txtfile: string);
  {-Create a .TXT file from DFM file}
  var
    InFile, OutFile: TFileStream;
  begin
    InFile := TFileStream.Create(dfmfile, fmOpenRead);
    OutFile := TFileStream.Create(txtfile, fmCreate);
    try
      ObjectResourceToText(InFile, OutFile);
    finally
      Infile.Free;
      OutFile.Free;
    end;
  end;

  procedure TSourceList.SetSourceFile(const Value: string);
  var
    FExt: string[4];
    DFMname: string;
  begin
    if Value <> FSourceFile then
    begin
      FSourceFile := Value;
      FSourcePath := ExtractFilePath(FSourceFile);
      { DFM file setup}
      if IsFormFile(FSourceFile) then
      begin
        FExt := ExtractFileExt(FSourceFile);
        if FExt[2] = '~' then
          FExt := '.~tx'
        else
          FExt := '.txt';
        DFMname := ChangeFileExt(FSourceFile, FExt);
        DFMtoTXT(FSourceFile, DFMname);
        FSourceFile := DFMname;
        FCreated := True;
      end
      else
        FCreated := False;
    end;
    Clear;
  end;

function TSourceList.DeleteCreatedFile: string;
{-Delete created .txt file}
var
  FN: string;
begin
  Result := '';
  if FCreated then
  begin
    FN := FSourceFile;
    if DeleteFile(FN) then
      Result := '  File '+FN+' deleted.'
    else
    begin
      MessageBeep(MB_ICONASTERISK);
      MessageDlg('  File '+FN+' was not deleted.', mtWarning, [mbOK], 0);
    end
  end
end;

end.
